const fs = require('fs-extra');
const path = require('path');
const { BrowserWindow } = require('electron');
const electron = require('electron');

const { download } = require('./electron-dl-quiet');
const Headset = require('./HeadsetSingleton');
const { EVENTS, SERVER_URL } = require('./consts');
const getLicense = require('./license');
const {
    STATUS_INSTALLING_FW,
    STATUS_DOWNLOADING_VERSION_INFO,
    STATUS_DOWNLOADING_FW,
    VERSION_BETA,
    VERSION_STABLE,
    HEADSET_STATUS
} = require('./consts');
const logger = require('../Logger').Logger;

class HeadsetWrapper {
    constructor(ipc) {
        this.ipc = ipc;
        this.win = null;
        this.serverVersionInfo = null;

        this.headset = Headset.getInstance();

        this.ipc.on(EVENTS.IS_DEVICE_CONNECTED, (event, _arg) => {
            let details = { connected: HEADSET_STATUS.HEADSET_DISCONNECTED };
            logger.info('headsetWrapper - IS_DEVICE_CONNECTED', event, _arg);
            if (this.headset.isConnected()) {
                try {
                    logger.info(
                        'HeadsetWrapper: IS_DEVICE_CONNECTED -  calling getHardwareModel'
                    );
                    details = this.headset.getHardwareModel();
                    if (!details) return;
                } catch (error) {
                    event.sender.send(`${EVENTS.FATAL_USB_ERROR}-push`, {});
                }

                details.connected = HEADSET_STATUS.HEADSET_CONNECTED;
                logger.info('Device is connected.');
            }

            details.isDfu = this.headset.isDfu();
            event.sender.send(
                `${EVENTS.IS_DEVICE_CONNECTED}-response`,
                details
            );
        });

        this.ipc.on(EVENTS.GET_DEVICE_DETAILS, (event, _arg) => {
            let details = null;
            if (this.headset.isConnected()) {
                try {
                    logger.info(
                        'HeadsetWrapper: GET_DEVICE_DETAILS -  calling getHardwareModel'
                    );
                    details = this.headset.getHardwareModel();
                    if (!details) return;
                } catch (error) {
                    event.sender.send(`${EVENTS.FATAL_USB_ERROR}-push`, {});
                }

                details.connected = HEADSET_STATUS.HEADSET_CONNECTED;
            }

            event.sender.send(`${EVENTS.GET_DEVICE_DETAILS}-response`, details);
        });

        this.ipc.on(EVENTS.FIRMWARE_UPDATE, (event, arg) => {
            this.win =
                this.win ||
                BrowserWindow.getFocusedWindow() ||
                BrowserWindow.getAllWindows()[0];
            const version = arg && arg.version ? arg.version : VERSION_STABLE;
            const language = arg && arg.language ? arg.language : null;
            this.downloadUpdate(
                event.sender,
                EVENTS.FIRMWARE_UPDATE,
                version,
                language
            );
        });

        this.ipc.on(EVENTS.RECOVERY_FIRMWARE_UPDATE, (event, _arg) => {
            this.win =
                this.win ||
                BrowserWindow.getFocusedWindow() ||
                BrowserWindow.getAllWindows()[0];
            if (this.headset.isDfu()) {
                // Real DFU - use last download FW file
                this.flashLastFW(event.sender, EVENTS.RECOVERY_FIRMWARE_UPDATE);
            } else {
                // Simulated DFU - re-download FW file
                this.downloadUpdate(
                    event.sender,
                    EVENTS.RECOVERY_FIRMWARE_UPDATE,
                    VERSION_STABLE
                );
            }
        });

        this.ipc.on(EVENTS.BATCH_FIRMWARE_UPDATE, (event, _arg) => {
            this.downloadUpdate(
                event.sender,
                EVENTS.BATCH_FIRMWARE_UPDATE,
                VERSION_STABLE
            );
        });

        this.ipc.on(EVENTS.IS_NEW_FIRMWARE, async event => {
            try {
                this.win =
                    this.win ||
                    BrowserWindow.getFocusedWindow() ||
                    BrowserWindow.getAllWindows()[0];
                this.serverVersionInfo = null;
                const result = await this.checkNewFirmware(event.sender);
                logger.info(
                    'IS_NEW_FIRMWARE response',
                    result,
                    JSON.stringify(result)
                );
                event.sender.send(`${EVENTS.IS_NEW_FIRMWARE}-response`, result);
            } catch (error) {
                logger.error('Error IS_NEW_FIRMWARE', error);
                logger.error(error);
                logger.error(JSON.stringify(error));
                event.sender.send(`${EVENTS.IS_NEW_FIRMWARE}-response`, {
                    error
                });
            }
        });
    }

    destroy() {
        this.headset.destroy();
    }

    async checkNewFirmware(sender, forceDownloadInDFU) {
        logger.info('HeadsetWrapper: checkNewFirmware ', forceDownloadInDFU);
        if (this.headset.flashing_fw) {
            logger.info('HeadsetWrapper: flashing_fw');
            return {};
        }

        if (!forceDownloadInDFU && !this.headset.isConnected()) {
            logger.info('HeadsetWrapper: not connected');
            throw Error('No headset connected');
        }

        let hw = {};
        logger.info(
            'HeadsetWrapper: checkNewFirmware -  calling getHardwareModel'
        );
        hw = this.headset.getHardwareModel();
        if (!forceDownloadInDFU) {
            hw.connected = HEADSET_STATUS.HEADSET_CONNECTED;
            logger.info(
                'HeadsetWrapper: checkNewFirmware -  sending IS_DEVICE_CONNECTED-push',
                hw
            );
            sender.send(`${EVENTS.IS_DEVICE_CONNECTED}-push`, hw);
        }

        const license = getLicense();

        const url = (license.serverUrl
            ? license.serverUrl
            : SERVER_URL
        ).replace(/{model}/, hw.model);
        logger.info('HeadsetWrapper: checking', url);

        if (!this.serverVersionInfo) {
            const jsonDir = electron.app.getPath('userData');
            const jsonPath = path.join(jsonDir, 'fw_info.json');

            await download(this.win, url, {
                directory: jsonDir,
                filename: 'fw_info.json'
            });
            this.serverVersionInfo = await fs.readJSON(jsonPath);
        }

        const { allowBeta, allowSuperUser } = license;

        const result = {
            currentVersion: hw.fw_version,
            stableVersion: this.serverVersionInfo.fw_version,
            betaVersion: this.serverVersionInfo.beta_version,
            betaBuildNumber: this.serverVersionInfo.beta_build_number,
            fw_url: this.serverVersionInfo.fw_url,
            hasBeta:
                typeof this.serverVersionInfo.beta_fw_url !== 'undefined' &&
                allowBeta,
            beta_fw_url:
                typeof this.serverVersionInfo.beta_fw_url !== 'undefined'
                    ? this.serverVersionInfo.beta_fw_url
                    : '',
            super_user_urls:
                typeof this.serverVersionInfo.super_user !== 'undefined' &&
                allowSuperUser
                    ? this.serverVersionInfo.super_user
                    : []
        };

        logger.info('HeadsetWrapper: results:', result);
        return result;
    }

    sendConnectionPush(sender, _error) {
        setTimeout(() => {
            let details = { connected: HEADSET_STATUS.HEADSET_DISCONNECTED };
            if (this.headset.isConnected()) {
                logger.info(
                    'HeadsetWrapper: sendConnectionPush -  calling getHardwareModel'
                );
                details = this.headset.getHardwareModel();
                if (!details) return;

                details.connected = HEADSET_STATUS.HEADSET_CONNECTED;
            }

            logger.info(
                'HeadsetWrapper: sendConnectionPush -  sending IS_DEVICE_CONNECTED-push',
                details
            );
            sender.send(`${EVENTS.IS_DEVICE_CONNECTED}-push`, details);
        }, 500);
    }

    async flashLastFW(sender, initiatingEvent) {
        try {
            const fwDir = electron.app.getPath('userData');
            const fwPath = path.join(fwDir, 'firmware.tar.gz');
            await this.headset.loadFW(fwPath, progress =>
                sender.send(`${initiatingEvent}-progress`, {
                    status: STATUS_INSTALLING_FW,
                    progress
                })
            );
            sender.send(`${initiatingEvent}-response`, { success: true });
            this.sendConnectionPush(sender);
        } catch (err) {
            logger.info(`flashLastFw - Error flashing image - ${err}`);
            const message = typeof err === 'string' ? err : err.message;
            sender.send(`${initiatingEvent}-response`, {
                success: false,
                error: message
            });
            this.sendConnectionPush(sender);
        }
    }

    async downloadUpdate(sender, initiatingEvent, version, language) {
        try {
            const forceDownloadInDFU =
                initiatingEvent === EVENTS.RECOVERY_FIRMWARE_UPDATE;
            const fwDir = electron.app.getPath('userData');

            sender.send(`${initiatingEvent}-progress`, {
                status: STATUS_DOWNLOADING_VERSION_INFO
            });

            const result = await this.checkNewFirmware(
                sender,
                forceDownloadInDFU
            );

            sender.send(`${initiatingEvent}-progress`, {
                status: STATUS_DOWNLOADING_FW,
                progress: 0
            });

            let url = '';
            switch (version) {
                case VERSION_BETA:
                    url = result.beta_fw_url;
                    break;
                case VERSION_STABLE:
                    url = result.fw_url;
                    break;
                default: {
                    const item = result.super_user_urls.filter(
                        itemUrl => itemUrl.fw_version === version
                    );
                    if (item.length > 0) {
                        url = item[0].fw_url; // eslint-disable-line
                    } else {
                        url = result.fw_url;
                    }
                }
            }

            let fwblob;

            try {
                if (this.headset.is_qcc) {
                    // QCC device - also fill in the language (so we'll burn the same language FW version)
                    url = url.replace(
                        '{language}',
                        language || this.headset.device_language
                    );
                }
                logger.info('Downloading FW URL', url);
                fwblob = await download(this.win, url, {
                    directory: fwDir,
                    filename: 'firmware.tar.gz',
                    timeout: 1000 * 60 * 3,
                    onProgress: progress => {
                        sender.send(`${initiatingEvent}-progress`, {
                            status: STATUS_DOWNLOADING_FW,
                            progress: 100 * progress
                        });

                        // Cancel download if headset is no longer connected
                        return this.headset.isConnected() || forceDownloadInDFU;
                    }
                });
            } catch (exc) {
                throw exc;
            }

            if (!forceDownloadInDFU && !this.headset.isConnected()) {
                throw Error('No headset connected');
            }

            sender.send(`${initiatingEvent}-progress`, {
                status: STATUS_INSTALLING_FW
            });

            const latestVersionPath = fwblob.getSavePath();
            const that = this;
            await this.headset.loadFW(latestVersionPath, progress => {
                if (!that.win.isDestroyed()) {
                    try {
                        sender.send(`${initiatingEvent}-progress`, {
                            status: STATUS_INSTALLING_FW,
                            progress
                        });

                        that.win.setProgressBar(progress / 100);
                    } catch (e) {
                        console.error(e);
                    }
                }
            });
            sender.send(`${initiatingEvent}-response`, { success: true });

            this.sendConnectionPush(sender);
        } catch (err) {
            const message = typeof err === 'string' ? err : err.message;
            logger.info(`FW update failed - ${message}`);
            sender.send(`${initiatingEvent}-response`, {
                success: false,
                error: message
            });
            this.sendConnectionPush(sender, true);
        }
    }
}

module.exports = HeadsetWrapper;
